<?php

declare(strict_types=1);

namespace Hexspoor\TranslatorClient;

use Hexspoor\Interoperability\Http\Header;
use Hexspoor\TranslatorClient\Contracts\ClientContract;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use function json_encode;
use function trim;

final class Client implements ClientContract
{
    private string $url;

    private string $token;

    public function __construct(string $url, string $token)
    {
        $this->url = $url;
        $this->token = $token;
    }

    public function postSalesOrder(
        int $connectorId,
        string $content,
        string $contentType = 'application/json'
    ): Response {
        return $this->post($connectorId, $content, $contentType, 'SalesOrder');
    }

    public function postStockValue(
        int $connectorId,
        string $content,
        string $contentType = 'application/json'
    ): Response {
        return $this->post($connectorId, $content, $contentType, 'StockValue');
    }

    public function post(int $connectorId, string $content, string $contentType, string $type): Response
    {
        $request = Http::withToken($this->token);

        return $request
            ->withHeaders([
                Header::X_CONNECTOR_ID => $connectorId,
                Header::X_MESSAGE_TYPE => $type,
            ])
            ->withBody($content, $contentType)
            ->post($this->formatUrl('translator/message'));
    }

    public function postShipment(
        int $connectorId,
        string $content,
        string $contentType = 'application/json'
    ): Response {
        return $this->post($connectorId, $content, $contentType, 'Shipment');
    }

    public function postProduct(
        int $connectorId,
        string $content,
        string $contentType = 'application/json'
    ): Response {
        return $this->post($connectorId, $content, $contentType, 'Product');
    }

    public function postIdentification(
        int $connectorId,
        int $messageId,
        string $identifier
    ): Response {
        $request = Http::withToken($this->token);

        $body = json_encode([
            'message_id' => $messageId,
            'identifier' => $identifier,
        ]);

        return $request
            ->withHeaders([
                Header::X_CONNECTOR_ID => $connectorId,
            ])
            ->withBody($body, 'application/json')
            ->post($this->formatUrl('translator/identification'));
    }

    private function formatUrl(string $url): string
    {
        $baseUrl = trim($this->url, '/');

        return "{$baseUrl}/{$url}";
    }
}
